-- InfoZone Forum - core schema (MySQL/MariaDB, utf8mb4)
SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS=0;

DROP TABLE IF EXISTS moderation_logs;
DROP TABLE IF EXISTS moderation_reports;
DROP TABLE IF EXISTS notifications;
DROP TABLE IF EXISTS private_messages;
DROP TABLE IF EXISTS attachments;
DROP TABLE IF EXISTS post_mentions;
DROP TABLE IF EXISTS posts;
DROP TABLE IF EXISTS topics;
DROP TABLE IF EXISTS forums;
DROP TABLE IF EXISTS categories;
DROP TABLE IF EXISTS sessions;
DROP TABLE IF EXISTS role_permissions;
DROP TABLE IF EXISTS user_roles;
DROP TABLE IF EXISTS permissions;
DROP TABLE IF EXISTS roles;
DROP TABLE IF EXISTS users;

CREATE TABLE users (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  username VARCHAR(40) NOT NULL,
  email VARCHAR(190) NOT NULL,
  pass_hash VARCHAR(255) NOT NULL,
  avatar_path VARCHAR(255) NULL,
  bio TEXT NULL,
  reputation INT NOT NULL DEFAULT 0,
  last_seen_at DATETIME NULL,
  is_active TINYINT(1) NOT NULL DEFAULT 1,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  UNIQUE KEY uq_users_username (username),
  UNIQUE KEY uq_users_email (email),
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE roles (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  name VARCHAR(80) NOT NULL,
  is_system TINYINT(1) NOT NULL DEFAULT 0,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  UNIQUE KEY uq_roles_name (name),
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE permissions (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  perm_key VARCHAR(120) NOT NULL,
  description VARCHAR(190) NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  UNIQUE KEY uq_permissions_key (perm_key),
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE user_roles (
  user_id BIGINT UNSIGNED NOT NULL,
  role_id BIGINT UNSIGNED NOT NULL,
  created_at DATETIME NOT NULL,
  PRIMARY KEY (user_id, role_id),
  CONSTRAINT fk_user_roles_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  CONSTRAINT fk_user_roles_role FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE role_permissions (
  role_id BIGINT UNSIGNED NOT NULL,
  permission_id BIGINT UNSIGNED NOT NULL,
  created_at DATETIME NOT NULL,
  PRIMARY KEY (role_id, permission_id),
  CONSTRAINT fk_role_permissions_role FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE,
  CONSTRAINT fk_role_permissions_perm FOREIGN KEY (permission_id) REFERENCES permissions(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE sessions (
  id VARCHAR(128) NOT NULL,
  user_id BIGINT UNSIGNED NULL,
  ip VARBINARY(16) NULL,
  user_agent VARCHAR(255) NULL,
  payload MEDIUMTEXT NOT NULL,
  last_activity_at DATETIME NOT NULL,
  created_at DATETIME NOT NULL,
  PRIMARY KEY (id),
  KEY idx_sessions_user (user_id),
  CONSTRAINT fk_sessions_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE categories (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  title VARCHAR(120) NOT NULL,
  slug VARCHAR(160) NOT NULL,
  description VARCHAR(255) NULL,
  sort_order INT NOT NULL DEFAULT 0,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  UNIQUE KEY uq_categories_slug (slug),
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE forums (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  category_id BIGINT UNSIGNED NOT NULL,
  title VARCHAR(120) NOT NULL,
  slug VARCHAR(160) NOT NULL,
  description VARCHAR(255) NULL,
  sort_order INT NOT NULL DEFAULT 0,
  is_locked TINYINT(1) NOT NULL DEFAULT 0,
  topic_count INT NOT NULL DEFAULT 0,
  post_count INT NOT NULL DEFAULT 0,
  last_post_at DATETIME NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  UNIQUE KEY uq_forums_slug (slug),
  KEY idx_forums_cat_order (category_id, sort_order),
  CONSTRAINT fk_forums_category FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE topics (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  forum_id BIGINT UNSIGNED NOT NULL,
  user_id BIGINT UNSIGNED NOT NULL,
  title VARCHAR(180) NOT NULL,
  slug VARCHAR(200) NOT NULL,
  is_locked TINYINT(1) NOT NULL DEFAULT 0,
  is_pinned TINYINT(1) NOT NULL DEFAULT 0,
  reply_count INT NOT NULL DEFAULT 0,
  view_count INT NOT NULL DEFAULT 0,
  smart_tags VARCHAR(255) NULL,
  ai_summary VARCHAR(600) NULL,
  mood ENUM('positive','neutral','negative','question','announcement') NOT NULL DEFAULT 'neutral',
  last_post_at DATETIME NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  UNIQUE KEY uq_topics_forum_slug (forum_id, slug),
  KEY idx_topics_forum_last (forum_id, last_post_at),
  CONSTRAINT fk_topics_forum FOREIGN KEY (forum_id) REFERENCES forums(id) ON DELETE CASCADE,
  CONSTRAINT fk_topics_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE RESTRICT,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE posts (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  topic_id BIGINT UNSIGNED NOT NULL,
  user_id BIGINT UNSIGNED NOT NULL,
  body_raw MEDIUMTEXT NOT NULL,
  body_html MEDIUMTEXT NOT NULL,
  mood ENUM('positive','neutral','negative','question','announcement') NOT NULL DEFAULT 'neutral',
  voice_path VARCHAR(255) NULL,
  voice_mime VARCHAR(80) NULL,
  voice_seconds INT NULL,
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  KEY idx_posts_topic_created (topic_id, created_at),
  CONSTRAINT fk_posts_topic FOREIGN KEY (topic_id) REFERENCES topics(id) ON DELETE CASCADE,
  CONSTRAINT fk_posts_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE RESTRICT,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE post_mentions (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  post_id BIGINT UNSIGNED NOT NULL,
  mentioned_user_id BIGINT UNSIGNED NOT NULL,
  created_at DATETIME NOT NULL,
  KEY idx_mentions_post (post_id),
  KEY idx_mentions_mentioned (mentioned_user_id),
  CONSTRAINT fk_mentions_post FOREIGN KEY (post_id) REFERENCES posts(id) ON DELETE CASCADE,
  CONSTRAINT fk_mentions_user FOREIGN KEY (mentioned_user_id) REFERENCES users(id) ON DELETE CASCADE,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE attachments (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  user_id BIGINT UNSIGNED NOT NULL,
  post_id BIGINT UNSIGNED NULL,
  original_name VARCHAR(255) NOT NULL,
  stored_path VARCHAR(255) NOT NULL,
  mime VARCHAR(120) NOT NULL,
  bytes BIGINT UNSIGNED NOT NULL,
  created_at DATETIME NOT NULL,
  KEY idx_attach_post (post_id),
  CONSTRAINT fk_attach_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE RESTRICT,
  CONSTRAINT fk_attach_post FOREIGN KEY (post_id) REFERENCES posts(id) ON DELETE SET NULL,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE private_messages (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  sender_id BIGINT UNSIGNED NOT NULL,
  recipient_id BIGINT UNSIGNED NOT NULL,
  subject VARCHAR(180) NOT NULL,
  body_raw MEDIUMTEXT NOT NULL,
  body_html MEDIUMTEXT NOT NULL,
  is_read TINYINT(1) NOT NULL DEFAULT 0,
  created_at DATETIME NOT NULL,
  KEY idx_pm_recipient (recipient_id, is_read),
  CONSTRAINT fk_pm_sender FOREIGN KEY (sender_id) REFERENCES users(id) ON DELETE RESTRICT,
  CONSTRAINT fk_pm_recipient FOREIGN KEY (recipient_id) REFERENCES users(id) ON DELETE RESTRICT,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE notifications (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  user_id BIGINT UNSIGNED NOT NULL,
  type VARCHAR(80) NOT NULL,
  payload JSON NOT NULL,
  is_read TINYINT(1) NOT NULL DEFAULT 0,
  created_at DATETIME NOT NULL,
  KEY idx_notifications_user (user_id, is_read, created_at),
  CONSTRAINT fk_notifications_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE moderation_reports (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  reporter_id BIGINT UNSIGNED NOT NULL,
  post_id BIGINT UNSIGNED NOT NULL,
  reason VARCHAR(190) NOT NULL,
  status ENUM('open','closed') NOT NULL DEFAULT 'open',
  created_at DATETIME NOT NULL,
  updated_at DATETIME NOT NULL,
  KEY idx_reports_status (status, created_at),
  CONSTRAINT fk_reports_reporter FOREIGN KEY (reporter_id) REFERENCES users(id) ON DELETE RESTRICT,
  CONSTRAINT fk_reports_post FOREIGN KEY (post_id) REFERENCES posts(id) ON DELETE CASCADE,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE moderation_logs (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  actor_id BIGINT UNSIGNED NOT NULL,
  action VARCHAR(120) NOT NULL,
  target_type VARCHAR(60) NOT NULL,
  target_id BIGINT UNSIGNED NOT NULL,
  payload JSON NULL,
  created_at DATETIME NOT NULL,
  KEY idx_mod_logs_actor (actor_id, created_at),
  CONSTRAINT fk_mod_logs_actor FOREIGN KEY (actor_id) REFERENCES users(id) ON DELETE RESTRICT,
  PRIMARY KEY (id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO roles (name, is_system, created_at, updated_at) VALUES
('Admin', 1, NOW(), NOW()),
('Moderator', 1, NOW(), NOW()),
('User', 1, NOW(), NOW());

INSERT INTO permissions (perm_key, description, created_at, updated_at) VALUES
('acp.access', 'Access the Admin Control Panel', NOW(), NOW()),
('forums.manage', 'Create/update/delete forums', NOW(), NOW()),
('topics.lock', 'Lock/unlock topics', NOW(), NOW()),
('posts.moderate', 'Hide/delete posts', NOW(), NOW());

INSERT INTO role_permissions (role_id, permission_id, created_at)
SELECT r.id, p.id, NOW()
FROM roles r
JOIN permissions p
WHERE r.name='Admin';

INSERT INTO role_permissions (role_id, permission_id, created_at)
SELECT r.id, p.id, NOW()
FROM roles r
JOIN permissions p
WHERE r.name='Moderator' AND p.perm_key IN ('acp.access','topics.lock','posts.moderate');

INSERT INTO categories (title, slug, description, sort_order, created_at, updated_at)
VALUES ('General', 'general', 'General discussion', 0, NOW(), NOW());

INSERT INTO forums (category_id, title, slug, description, sort_order, is_locked, topic_count, post_count, last_post_at, created_at, updated_at)
VALUES (1, 'Welcome to InfoZone', 'welcome', 'Announcements and introductions', 0, 0, 1, 1, NOW(), NOW(), NOW());

INSERT INTO users (username, email, pass_hash, created_at, updated_at, is_active)
VALUES ('admin', 'admin@infozonelive.com', '$2y$10$2y5FQzSg5e1Cw2oD0WlO6u3WcJrj8Ssn5VQdY4p8cT3r2l9GZtK3u', NOW(), NOW(), 1);

-- admin password is: admin123 (change immediately)
INSERT INTO user_roles (user_id, role_id, created_at)
SELECT 1, r.id, NOW() FROM roles r WHERE r.name='Admin';

INSERT INTO topics (forum_id, user_id, title, slug, smart_tags, ai_summary, mood, last_post_at, created_at, updated_at)
VALUES (1, 1, 'Start here: rules and how to post', 'start-here-rules', 'welcome,rules', 'A short summary will live here later.', 'announcement', NOW(), NOW(), NOW());

INSERT INTO posts (topic_id, user_id, body_raw, body_html, mood, created_at, updated_at)
VALUES (1, 1, 'Welcome to InfoZone Forum!\n\nThis is a starter seed post.', 'Welcome to InfoZone Forum!<br><br>This is a starter seed post.', 'positive', NOW(), NOW());

SET FOREIGN_KEY_CHECKS=1;
