<?php
declare(strict_types=1);

namespace InfoZone\Controller\Acp;

use InfoZone\Http\Request;
use InfoZone\Http\Response;
use InfoZone\Support\View;
use InfoZone\Support\Csrf;
use InfoZone\Security\Ability;
use InfoZone\Repository\ForumRepository;
use InfoZone\Core\Database;

final class ForumAdminController
{
    public function index(Request $req, array $params): Response
    {
        Ability::require('forums.manage');

        $db = Database::pdo();
        $categories = $db->query("SELECT * FROM categories ORDER BY sort_order ASC")->fetchAll();
        $forums = $db->query("SELECT * FROM forums ORDER BY category_id ASC, sort_order ASC")->fetchAll();

        $html = View::render('acp/forums.php', [
            'categories' => $categories,
            'forums' => $forums,
            'pageTitle' => 'ACP • Forums',
        ]);

        return Response::html($html);
    }

    public function create(Request $req, array $params): Response
    {
        Ability::require('forums.manage');
        Csrf::verify($req->post);

        $data = $this->validateForumPayload($req->post);
        (new ForumRepository())->createForum($data);

        return Response::redirect('/acp/forums');
    }

    public function update(Request $req, array $params): Response
    {
        Ability::require('forums.manage');
        Csrf::verify($req->post);

        $id = (int)($req->post['id'] ?? 0);
        if ($id <= 0) return Response::html('Invalid forum id.', 422);

        $data = $this->validateForumPayload($req->post);
        (new ForumRepository())->updateForum($id, $data);

        return Response::redirect('/acp/forums');
    }

    public function delete(Request $req, array $params): Response
    {
        Ability::require('forums.manage');
        Csrf::verify($req->post);

        $id = (int)($req->post['id'] ?? 0);
        if ($id <= 0) return Response::html('Invalid forum id.', 422);

        (new ForumRepository())->deleteForum($id);
        return Response::redirect('/acp/forums');
    }

    public function reorder(Request $req, array $params): Response
    {
        Ability::require('forums.manage');
        Csrf::verify($req->post);

        $orders = $req->post['orders'] ?? [];
        if (!is_array($orders)) $orders = [];

        (new ForumRepository())->reorderForums($orders);
        return Response::redirect('/acp/forums');
    }

    private function validateForumPayload(array $post): array
    {
        $categoryId = (int)($post['category_id'] ?? 0);
        $title = trim((string)($post['title'] ?? ''));
        $slug = trim((string)($post['slug'] ?? ''));
        $description = trim((string)($post['description'] ?? ''));
        $sortOrder = (int)($post['sort_order'] ?? 0);
        $isLocked = isset($post['is_locked']) ? 1 : 0;

        if ($categoryId <= 0 || $title === '' || mb_strlen($title) > 120) {
            http_response_code(422);
            echo 'Invalid forum payload.';
            exit;
        }
        if ($slug === '') $slug = $this->slugify($title);
        if (!preg_match('/^[a-z0-9-]{1,160}$/', $slug)) {
            http_response_code(422);
            echo 'Invalid slug.';
            exit;
        }

        return [
            'category_id' => $categoryId,
            'title' => $title,
            'slug' => $slug,
            'description' => $description,
            'sort_order' => $sortOrder,
            'is_locked' => $isLocked,
        ];
    }

    private function slugify(string $s): string
    {
        $s = mb_strtolower($s);
        $s = preg_replace('/[^a-z0-9\s-]/', '', $s) ?? '';
        $s = preg_replace('/\s+/', '-', trim($s)) ?? '';
        $s = preg_replace('/-+/', '-', $s) ?? '';
        return $s ?: 'forum';
    }
}
