<?php
declare(strict_types=1);

namespace InfoZone\Controller;

use InfoZone\Http\Request;
use InfoZone\Http\Response;
use InfoZone\Repository\TopicRepository;
use InfoZone\Support\View;
use InfoZone\Support\Csrf;
use InfoZone\Security\Auth;

final class TopicController
{
    public function viewTopic(Request $req, array $params): Response
    {
        $id = (int)($params['id'] ?? 0);
        $repo = new TopicRepository();
        $topic = $repo->findTopic($id);

        if (!$topic) return Response::html(View::render('errors/404.php', ['pageTitle' => 'Not Found']), 404);

        $posts = $repo->listPosts($id);
        $related = $repo->relatedTopics((int)$topic['forum_id'], $id);

        $html = View::render('topic/view.php', [
            'topic' => $topic,
            'posts' => $posts,
            'related' => $related,
            'pageTitle' => $topic['title'],
            'metaDescription' => $topic['ai_summary'] ?: ('Discussion: ' . $topic['title']),
        ]);

        return Response::html($html);
    }

    public function addReply(Request $req, array $params): Response
    {
        Auth::requireLogin();
        Csrf::verify($req->post);

        $topicId = (int)($params['id'] ?? 0);
        $body = trim((string)($req->post['body'] ?? ''));
        $mood = (string)($req->post['mood'] ?? 'neutral');

        $allowedMoods = ['positive','neutral','negative','question','announcement'];
        if (!in_array($mood, $allowedMoods, true)) $mood = 'neutral';

        if ($topicId <= 0 || $body === '' || mb_strlen($body) > 20000) {
            return Response::html('Invalid input.', 422);
        }

        (new TopicRepository())->addReply($topicId, (int)Auth::userId(), $body, $mood);
        return Response::redirect('/topic/' . $topicId);
    }
}
