<?php
declare(strict_types=1);

namespace InfoZone\Core;

use Dotenv\Dotenv;
use InfoZone\Http\Router;
use InfoZone\Http\Request;
use InfoZone\Http\Response;
use InfoZone\Support\View;
use InfoZone\Support\Csrf;

final class App
{
    private string $basePath;
    private array $config = [];
    private Router $router;

    private function __construct(string $basePath)
    {
        $this->basePath = rtrim($basePath, '/');
        $this->router = new Router();
    }

    public static function bootstrap(string $basePath): self
    {
        $app = new self($basePath);

        // Load .env
        if (file_exists($app->basePath . '/.env')) {
            Dotenv::createImmutable($app->basePath)->safeLoad();
        } else {
            Dotenv::createImmutable($app->basePath, '.env.example')->safeLoad();
        }

        $app->config['app'] = require $app->basePath . '/config/app.php';
        $app->config['db']  = require $app->basePath . '/config/database.php';

        date_default_timezone_set($app->config['app']['timezone']);

        // Session
        $sessionName = $_ENV['SESSION_NAME'] ?? 'infozone_session';
        session_name($sessionName);
        if (session_status() !== PHP_SESSION_ACTIVE) {
            session_start();
        }

        // Core services
        Database::init($app->config['db']);
        View::init($app->basePath . '/templates', $app->config['app']);
        Csrf::init($_ENV['CSRF_TOKEN_KEY'] ?? '_csrf');

        $app->registerRoutes();
        return $app;
    }

    private function registerRoutes(): void
    {
        // Public
        $this->router->get('/', [\InfoZone\Controller\ForumController::class, 'index']);
        $this->router->get('/forum/{id}', [\InfoZone\Controller\ForumController::class, 'viewForum']);
        $this->router->get('/topic/{id}', [\InfoZone\Controller\TopicController::class, 'viewTopic']);
        $this->router->post('/topic/{id}/reply', [\InfoZone\Controller\TopicController::class, 'addReply']);

        // ACP
        $this->router->get('/acp', [\InfoZone\Controller\Acp\DashboardController::class, 'index']);
        $this->router->get('/acp/forums', [\InfoZone\Controller\Acp\ForumAdminController::class, 'index']);
        $this->router->post('/acp/forums/create', [\InfoZone\Controller\Acp\ForumAdminController::class, 'create']);
        $this->router->post('/acp/forums/update', [\InfoZone\Controller\Acp\ForumAdminController::class, 'update']);
        $this->router->post('/acp/forums/delete', [\InfoZone\Controller\Acp\ForumAdminController::class, 'delete']);
        $this->router->post('/acp/forums/reorder', [\InfoZone\Controller\Acp\ForumAdminController::class, 'reorder']);
    }

    public function run(): void
    {
        $request = Request::fromGlobals();
        $response = $this->router->dispatch($request);

        if ($response instanceof Response) {
            $response->send();
            return;
        }

        Response::html('Not Found', 404)->send();
    }
}
