<?php
declare(strict_types=1);

namespace InfoZone\Repository;

final class ForumRepository extends BaseRepository
{
    public function listCategoriesWithForums(): array
    {
        $cats = $this->db->query("SELECT * FROM categories ORDER BY sort_order ASC, id ASC")->fetchAll();
        $forums = $this->db->query("SELECT * FROM forums ORDER BY category_id ASC, sort_order ASC, id ASC")->fetchAll();

        $byCat = [];
        foreach ($cats as $c) $byCat[(int)$c['id']] = $c + ['forums' => []];
        foreach ($forums as $f) {
            $cid = (int)$f['category_id'];
            if (isset($byCat[$cid])) $byCat[$cid]['forums'][] = $f;
        }
        return array_values($byCat);
    }

    public function findForum(int $id): ?array
    {
        $st = $this->db->prepare("SELECT * FROM forums WHERE id = ?");
        $st->execute([$id]);
        $row = $st->fetch();
        return $row ?: null;
    }

    public function listTopics(int $forumId, int $limit = 30): array
    {
        $st = $this->db->prepare("
            SELECT t.*, u.username AS author_name
            FROM topics t
            JOIN users u ON u.id = t.user_id
            WHERE t.forum_id = ?
            ORDER BY t.is_pinned DESC, t.last_post_at DESC
            LIMIT {$limit}
        ");
        $st->execute([$forumId]);
        return $st->fetchAll();
    }

    public function createForum(array $data): int
    {
        $st = $this->db->prepare("
            INSERT INTO forums (category_id, title, slug, description, sort_order, is_locked, created_at, updated_at)
            VALUES (:category_id, :title, :slug, :description, :sort_order, :is_locked, NOW(), NOW())
        ");
        $st->execute([
            ':category_id' => (int)$data['category_id'],
            ':title' => $data['title'],
            ':slug' => $data['slug'],
            ':description' => $data['description'],
            ':sort_order' => (int)$data['sort_order'],
            ':is_locked' => (int)$data['is_locked'],
        ]);
        return (int)$this->db->lastInsertId();
    }

    public function updateForum(int $id, array $data): void
    {
        $st = $this->db->prepare("
            UPDATE forums
            SET category_id=:category_id, title=:title, slug=:slug, description=:description,
                sort_order=:sort_order, is_locked=:is_locked, updated_at=NOW()
            WHERE id=:id
        ");
        $st->execute([
            ':id' => $id,
            ':category_id' => (int)$data['category_id'],
            ':title' => $data['title'],
            ':slug' => $data['slug'],
            ':description' => $data['description'],
            ':sort_order' => (int)$data['sort_order'],
            ':is_locked' => (int)$data['is_locked'],
        ]);
    }

    public function deleteForum(int $id): void
    {
        $st = $this->db->prepare("DELETE FROM forums WHERE id = ?");
        $st->execute([$id]);
    }

    public function reorderForums(array $orders): void
    {
        $st = $this->db->prepare("UPDATE forums SET sort_order = :sort_order WHERE id = :id");
        foreach ($orders as $id => $order) {
            $st->execute([':id' => (int)$id, ':sort_order' => (int)$order]);
        }
    }
}
