<?php
declare(strict_types=1);

namespace InfoZone\Repository;

final class TopicRepository extends BaseRepository
{
    public function findTopic(int $id): ?array
    {
        $st = $this->db->prepare("
            SELECT t.*, u.username AS author_name, f.title AS forum_title
            FROM topics t
            JOIN users u ON u.id = t.user_id
            JOIN forums f ON f.id = t.forum_id
            WHERE t.id = ?
        ");
        $st->execute([$id]);
        $row = $st->fetch();
        return $row ?: null;
    }

    public function listPosts(int $topicId, int $limit = 50): array
    {
        $st = $this->db->prepare("
            SELECT p.*, u.username, u.avatar_path
            FROM posts p
            JOIN users u ON u.id = p.user_id
            WHERE p.topic_id = ?
            ORDER BY p.created_at ASC
            LIMIT {$limit}
        ");
        $st->execute([$topicId]);
        return $st->fetchAll();
    }

    public function addReply(int $topicId, int $userId, string $rawBody, string $mood = 'neutral'): int
    {
        $safe = nl2br(htmlspecialchars($rawBody, ENT_QUOTES, 'UTF-8'));

        $st = $this->db->prepare("
            INSERT INTO posts (topic_id, user_id, body_raw, body_html, mood, created_at, updated_at)
            VALUES (:topic_id, :user_id, :body_raw, :body_html, :mood, NOW(), NOW())
        ");
        $st->execute([
            ':topic_id' => $topicId,
            ':user_id' => $userId,
            ':body_raw' => $rawBody,
            ':body_html' => $safe,
            ':mood' => $mood,
        ]);
        $postId = (int)$this->db->lastInsertId();

        $this->db->prepare("
            UPDATE topics
            SET last_post_at = NOW(),
                reply_count = (SELECT COUNT(*) - 1 FROM posts WHERE topic_id = :topic_id),
                updated_at = NOW()
            WHERE id = :topic_id
        ")->execute([':topic_id' => $topicId]);

        return $postId;
    }

    public function relatedTopics(int $forumId, int $topicId, int $limit = 5): array
    {
        $st = $this->db->prepare("
            SELECT id, title, slug, last_post_at
            FROM topics
            WHERE forum_id = :forum_id AND id <> :topic_id
            ORDER BY last_post_at DESC
            LIMIT {$limit}
        ");
        $st->execute([':forum_id' => $forumId, ':topic_id' => $topicId]);
        return $st->fetchAll();
    }
}
